<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\Consignment;
use App\Models\ConsignmentItem;
use App\Models\Cliente;
use App\Models\Produto;
use App\Models\Estoque;
use App\Models\ProdutoVariacao;
use App\Models\Empresa;
use App\Models\VendaCaixa;
use App\Models\ItemVendaCaixa;
use Illuminate\Support\Facades\DB;
use Carbon\Carbon;

class ConsignmentController extends Controller
{
    public function index(Request $request)
    {
        $titulo = 'Consignações';
        $empresa_id = session('empresa_id');

        $query = Consignment::where('empresa_id', $empresa_id)
            ->with(['cliente', 'items', 'items.produto', 'items.produtoVariacao']);
        
        // Aplicar filtros se existirem
        if ($request->has('cliente_id') && $request->cliente_id != '') {
            $query->where('cliente_id', $request->cliente_id);
        }
        
        if ($request->has('status') && $request->status != '') {
            $query->where('status', $request->status);
        }
        
        if ($request->has('data_inicio') && $request->data_inicio != '') {
            $query->whereDate('data_abertura', '>=', $request->data_inicio);
        }
        
        if ($request->has('data_fim') && $request->data_fim != '') {
            $query->whereDate('data_abertura', '<=', $request->data_fim);
        }
        
        $consignments = $query->orderBy('created_at', 'desc')->paginate(20);
        $clientes = Cliente::where('empresa_id', $empresa_id)->orderBy('razao_social')->get();
        
        return view('consignments.index', compact('titulo', 'consignments', 'clientes'));
    }

    public function create()
    {
        \Illuminate\Support\Facades\Log::info('--- DEBUG create (ConsignmentController) INICIO ---');
        \Illuminate\Support\Facades\Log::info('Session empresa_id: ' . session('empresa_id'));
        $titulo = 'Nova Consignação';
        $empresa_id = session('empresa_id');
        $clientes = \App\Models\Cliente::where('empresa_id', $empresa_id)->orderBy('razao_social')->get();
        
        \Illuminate\Support\Facades\Log::info('create: Clientes encontrados: ' . $clientes->count());
        \Illuminate\Support\Facades\Log::info('--- DEBUG create (ConsignmentController) FIM ---');
        $empresa_id = session('empresa_id'); // Garante que temos o empresa_id aqui

        $selectable_products = [];
        $company_products = \App\Models\Produto::where('empresa_id', $empresa_id)
            ->with(['variacoes'])
            ->orderBy('nome')
            ->get();

        foreach ($company_products as $product) {
            if ($product->variacoes && $product->variacoes->count() > 0) {
                foreach ($product->variacoes as $variation) {
                    $stock = $product->estoqueAtualVariacao($variation->id);
                    $current_stock = $stock ? $stock->quantidade : 0;
                    if ($current_stock > 0) {
                        $selectable_products[] = [
                            'id_attr' => 'v_' . $variation->id,
                            'display_text' => $product->nome . ' - ' . $variation->descricao,
                            'produto_id' => $product->id,
                            'variacao_id' => $variation->id,
                            'nome_produto' => $product->nome,
                            'nome_variacao' => $variation->descricao,
                            'valor' => $variation->valor_venda > 0 ? $variation->valor_venda : $product->valor_venda, // Prioriza valor da variação
                            'estoque_disponivel' => $current_stock
                        ];
                    }
                }
            } else {
                $stock = $product->estoqueAtual();
                $current_stock = $stock ? $stock->quantidade : 0;
                if ($current_stock > 0) {
                    $selectable_products[] = [
                        'id_attr' => 'p_' . $product->id,
                        'display_text' => $product->nome,
                        'produto_id' => $product->id,
                        'variacao_id' => null,
                        'nome_produto' => $product->nome,
                        'nome_variacao' => null,
                        'valor' => $product->valor_venda,
                        'estoque_disponivel' => $current_stock
                    ];
                }
            }
        }

        
        \Illuminate\Support\Facades\Log::info('create: Produtos selecionáveis encontrados: ' . count($selectable_products));
        \Illuminate\Support\Facades\Log::info('--- DEBUG create (ConsignmentController) FIM ---');
        return view('consignments.create', compact('titulo', 'clientes', 'selectable_products'));
    }

    public function store(Request $request)
    {
        $request->validate([
            'cliente_id' => 'required',
            'data_abertura' => 'required|date',
            'observacao' => 'nullable|string',
            'produtos' => 'required|array',
            'produtos.*.id' => 'required|exists:produtos,id',
            'produtos.*.variacao_id' => 'nullable|exists:produto_variacoes,id',
            'produtos.*.quantidade' => 'required|numeric|min:0.001',
            'produtos.*.valor_unitario' => 'required|numeric|min:0',
        ]);

        DB::beginTransaction();
        
        try {
            $consignment = Consignment::create([
                'cliente_id' => $request->cliente_id,
                'empresa_id' => session('empresa_id'),
                'observacao' => $request->observacao,
                'data_abertura' => $request->data_abertura,
                'status' => 'aberto',
            ]);

            foreach ($request->produtos as $produto) {
                // Criar item de consignação
                $item = new ConsignmentItem([
                    'produto_id' => $produto['id'],
                    'produto_variacao_id' => $produto['variacao_id'] ?: null,
                    'quantidade_consignada' => $produto['quantidade'],
                    'quantidade_devolvida' => 0,
                    'quantidade_vendida' => 0,
                    'valor_unitario' => $produto['valor_unitario'],
                ]);
                
                $consignment->items()->save($item);
                
                // Atualizar estoque (reduzir estoque ao criar consignação)
                if ($produto['variacao_id']) {
                    $estoque = Estoque::where('produto_id', $produto['id'])
                        ->where('produto_variacao_id', $produto['variacao_id'])
                        ->first();
                } else {
                    $estoque = Estoque::where('produto_id', $produto['id'])
                        ->whereNull('produto_variacao_id')
                        ->first();
                }
                
                if ($estoque) {
                    $estoque->quantidade -= $produto['quantidade'];
                    $estoque->save();
                }
            }
            
            DB::commit();
            
            return redirect()
                ->route('consignments.index')
                ->with('success', 'Consignação criada com sucesso!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Erro ao criar consignação: ' . $e->getMessage());
        }
    }

    public function show($id)
    {
        $titulo = 'Detalhes da Consignação';
        $consignment = Consignment::with([
            'cliente', 
            'items', 
            'items.produto', 
            'items.produtoVariacao'
        ])->findOrFail($id);
        
        return view('consignments.show', compact('titulo', 'consignment'));
    }

    public function edit($id)
    {
        $titulo = 'Editar Consignação';
        $consignment = Consignment::with([
            'cliente', 
            'items', 
            'items.produto', 
            'items.produtoVariacao'
        ])->findOrFail($id);
        
        if ($consignment->status !== 'aberto') {
            return redirect()
                ->route('consignments.index')
                ->with('error', 'Apenas consignações abertas podem ser editadas.');
        }
        
        return view('consignments.edit', compact('titulo', 'consignment'));
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'observacao' => 'nullable|string',
        ]);

        $consignment = Consignment::findOrFail($id);
        
        if ($consignment->status !== 'aberto') {
            return redirect()
                ->route('consignments.index')
                ->with('error', 'Apenas consignações abertas podem ser editadas.');
        }
        
        $consignment->update([
            'observacao' => $request->observacao,
        ]);
        
        return redirect()
            ->route('consignments.index')
            ->with('success', 'Consignação atualizada com sucesso!');
    }

    public function finalize($id)
    {
        $titulo = 'Finalizar Consignação';
        $consignment = Consignment::with([
            'cliente', 
            'items', 
            'items.produto', 
            'items.produtoVariacao'
        ])->findOrFail($id);
        
        if ($consignment->status !== 'aberto') {
            return redirect()
                ->route('consignments.index')
                ->with('error', 'Apenas consignações abertas podem ser finalizadas.');
        }
        
        return view('consignments.finalize', compact('titulo', 'consignment'));
    }

    public function finalizeProcess(Request $request, $id)
    {
        $request->validate([
            'items' => 'required|array',
            'items.*.id' => 'required|exists:consignment_items,id',
            'items.*.quantidade_devolvida' => 'required|numeric|min:0',
        ]);

        $consignment = Consignment::with('items')->findOrFail($id);
        
        if ($consignment->status !== 'aberto') {
            return redirect()
                ->route('consignments.index')
                ->with('error', 'Apenas consignações abertas podem ser finalizadas.');
        }
        
        DB::beginTransaction();
        
        try {
            // Criar uma venda para os itens não devolvidos (vendidos)
            $venda = new VendaCaixa();
            $venda->cliente_id = $consignment->cliente_id;
            $venda->empresa_id = $consignment->empresa_id;
            $venda->usuario_id = auth()->user()->id;
            $venda->valor_total = 0; // Será atualizado mais tarde
            $venda->desconto = 0;
            $venda->acrescimo = 0;
            $venda->tipo_pagamento = 'credito'; // Pode ser configurável
            $venda->estado = 'APROVADO';
            $venda->NFe = 0;
            $venda->observacao = 'Venda gerada a partir da consignação #' . $consignment->id;
            $venda->data_registro = Carbon::now();
            $venda->save();
            
            $total_venda = 0;
            
            foreach ($request->items as $itemData) {
                $item = ConsignmentItem::find($itemData['id']);
                
                if (!$item || $item->consignment_id != $consignment->id) {
                    continue;
                }
                
                $quantidade_devolvida = (float) $itemData['quantidade_devolvida'];
                
                // Se a quantidade a ser devolvida for maior que a quantidade consignada, ajustar
                if ($quantidade_devolvida > $item->quantidade_consignada) {
                    $quantidade_devolvida = $item->quantidade_consignada;
                }
                
                // Calcular quantidade vendida
                $quantidade_vendida = $item->quantidade_consignada - $quantidade_devolvida;
                
                // Atualizar item da consignação
                $item->quantidade_devolvida = $quantidade_devolvida;
                $item->quantidade_vendida = $quantidade_vendida;
                $item->save();
                
                // Devolver ao estoque os itens devolvidos
                if ($quantidade_devolvida > 0) {
                    if ($item->produto_variacao_id) {
                        $estoque = Estoque::firstOrCreate(
                            [
                                'produto_id' => $item->produto_id,
                                'produto_variacao_id' => $item->produto_variacao_id,
                            ],
                            ['quantidade' => 0]
                        );
                    } else {
                        $estoque = Estoque::firstOrCreate(
                            [
                                'produto_id' => $item->produto_id,
                                'produto_variacao_id' => null,
                            ],
                            ['quantidade' => 0]
                        );
                    }
                    
                    $estoque->quantidade += $quantidade_devolvida;
                    $estoque->save();
                }
                
                // Adicionar à venda os itens vendidos
                if ($quantidade_vendida > 0) {
                    $itemVenda = new ItemVendaCaixa();
                    $itemVenda->venda_caixa_id = $venda->id;
                    $itemVenda->produto_id = $item->produto_id;
                    $itemVenda->quantidade = $quantidade_vendida;
                    $itemVenda->valor = $item->valor_unitario;
                    $itemVenda->observacao = 'Item da consignação #' . $consignment->id;
                    $itemVenda->save();
                    
                    // Calcular o total da venda
                    $total_venda += $quantidade_vendida * $item->valor_unitario;
                }
            }
            
            // Atualizar o total da venda
            $venda->valor_total = $total_venda;
            $venda->save();
            
            // Finalizar consignação
            $consignment->status = 'concluido';
            $consignment->data_fechamento = Carbon::now();
            $consignment->save();
            
            DB::commit();
            
            return redirect()
                ->route('consignments.index')
                ->with('success', 'Consignação finalizada com sucesso! Venda #' . $venda->id . ' gerada.');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Erro ao finalizar consignação: ' . $e->getMessage());
        }
    }

    public function cancel($id)
    {
        $consignment = Consignment::with('items')->findOrFail($id);
        
        if ($consignment->status !== 'aberto') {
            return redirect()
                ->route('consignments.index')
                ->with('error', 'Apenas consignações abertas podem ser canceladas.');
        }
        
        DB::beginTransaction();
        
        try {
            // Devolver todos os itens ao estoque
            foreach ($consignment->items as $item) {
                if ($item->produto_variacao_id) {
                    $estoque = Estoque::firstOrCreate(
                        [
                            'produto_id' => $item->produto_id,
                            'produto_variacao_id' => $item->produto_variacao_id,
                        ],
                        ['quantidade' => 0]
                    );
                } else {
                    $estoque = Estoque::firstOrCreate(
                        [
                            'produto_id' => $item->produto_id,
                            'produto_variacao_id' => null,
                        ],
                        ['quantidade' => 0]
                    );
                }
                
                $estoque->quantidade += $item->quantidade_consignada;
                $estoque->save();
            }
            
            // Cancelar consignação
            $consignment->status = 'cancelado';
            $consignment->data_fechamento = Carbon::now();
            $consignment->save();
            
            DB::commit();
            
            return redirect()
                ->route('consignments.index')
                ->with('success', 'Consignação cancelada com sucesso!');
                
        } catch (\Exception $e) {
            DB::rollBack();
            return redirect()
                ->back()
                ->withInput()
                ->with('error', 'Erro ao cancelar consignação: ' . $e->getMessage());
        }
    }

    /*
    public function searchProducts(Request $request)
    {
        \Illuminate\Support\Facades\Log::info('--- DEBUG searchProducts INICIO ---');
        \Illuminate\Support\Facades\Log::info('Request All: ' . json_encode($request->all()));
        \Illuminate\Support\Facades\Log::info('Request term: ' . $request->input('term'));
        \Illuminate\Support\Facades\Log::info('Session empresa_id: ' . session('empresa_id'));
        $empresa_id = session('empresa_id');
        $query = $request->get('term');
        
        $produtos = Produto::where('empresa_id', $empresa_id)
            ->where(function($q) use ($query) {
                $q->where('nome', 'LIKE', "%{$query}%")
                  ->orWhere('codigo_barras', 'LIKE', "%{$query}%")
                  ->orWhere('referencia', 'LIKE', "%{$query}%");
            })
            ->with(['estoque', 'variacoes'])
            ->get();
            
        $results = [];
        
        foreach ($produtos as $produto) {
            // Se o produto não tem variações, adicionar diretamente
            if ($produto->variacoes->count() == 0) {
                $estoque = $produto->estoque ? $produto->estoque->quantidade : 0;
                
                if ($estoque > 0) {
                    $results[] = [
                        'id' => $produto->id,
                        'nome' => $produto->nome,
                        'estoque' => $estoque,
                        'valor' => $produto->valor_venda ?? $produto->valor_unitario,
                        'variacao_id' => null,
                        'variacao_nome' => null
                    ];
                }
            } else {
                // Se tem variações, adicionar cada uma
                foreach ($produto->variacoes as $variacao) {
                    $estoqueVar = Estoque::where('produto_id', $produto->id)
                        ->where('produto_variacao_id', $variacao->id)
                        ->first();
                    
                    $quantidadeEstoque = $estoqueVar ? $estoqueVar->quantidade : 0;
                    
                    if ($quantidadeEstoque > 0) {
                        $results[] = [
                            'id' => $produto->id,
                            'nome' => $produto->nome,
                            'estoque' => $quantidadeEstoque,
                            'valor' => $variacao->valor ?? $produto->valor_venda ?? $produto->valor_unitario,
                            'variacao_id' => $variacao->id,
                            'variacao_nome' => $variacao->descricao ?? ''
                        ];
                    }
                }
            }
        }
        
        \Illuminate\Support\Facades\Log::info('searchProducts: Resultados encontrados: ' . count($results));
        \Illuminate\Support\Facades\Log::info('searchProducts: Resultados (JSON): ' . json_encode($results));
        \Illuminate\Support\Facades\Log::info('--- DEBUG searchProducts FIM ---');
        return response()->json($results);
    }
    */
}
