<?php

namespace App\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Illuminate\Support\Facades\Log;

class EvolutionAPIService
{
    protected $url_base;
    protected $global_key;

    public function __construct()
    {
        $this->url_base = env('EVOLUTION_APIURL');
        $this->global_key = env('EVOLUTION_GLOBALKEY');
    }

    public function createInstance($name, $number = '')
    {
        $client = $this->guzzle();

        try {
            $jsonResponse = [
                'instanceName' => $name,
                'token' => uniqid(),
                'number' => $number,
                'integration' => 'WHATSAPP-BAILEYS',
                'groupsIgnore' => true,
                'syncFullHistory' => false
            ];

            if (empty($number)) {
                unset($jsonResponse['number']);
            }

            $response = $client->post($this->url_base . '/instance/create', [
                'json' => $jsonResponse
            ]);

            $responseBody = json_decode($response->getBody()->getContents(), true);
            return $responseBody;
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function removeInstance($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->delete($this->url_base . '/instance/delete/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function instanceCheck($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->get($this->url_base . '/instance/connectionState/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            if ($e->getCode() == 404) {
                return [
                    'status' => 404,
                    'message' => 'Instance not found'
                ];
            }
            return $this->handleException($e);
        }
    }

    public function instances($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->get($this->url_base . '/instance/fetchInstances?instanceName=' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            // Obter o corpo da resposta da exceção
            $responseBody = $e->hasResponse() ? $e->getResponse()->getBody()->getContents() : null;

            // Retornar o corpo da resposta da exceção em JSON
            return [
                'error' => 'RequestException',
                'message' => $e->getMessage(),
                'response' => json_decode($responseBody, true)
            ];
        }
    }

    public function instanceConnect($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->get($this->url_base . '/instance/connect/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function instanceLogout($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->delete($this->url_base . '/instance/logout/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function fetchProfile($name, $number)
    {
        $client = $this->guzzle();

        try {
            $response = $client->post($this->url_base . '/chat/fetchProfile/' . $name, [
                'json' => [
                    'number' => $this->formatNumber($number)
                ]
            ]);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function call($name, $number)
    {
        $client = $this->guzzle();

        try {
            $response = $client->post($this->url_base . '/call/offer/' . $name, [
                'json' => [
                    'number' => $this->formatNumber($number),
                    'isVideo' => false,
                    'callDuration' => 3
                ]
            ]);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function restartInstance($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->post($this->url_base . '/instance/restart/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function updatePhotoProfile($name, $picture)
    {
        $client = $this->guzzle();

        try {
            $response = $client->put($this->url_base . '/chat/updateProfilePicture/' . $name, [
                'json' => [
                    'picture' => $picture
                ]
            ]);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function chatConversations($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->post($this->url_base . '/chat/findChats/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function findContacts($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->post($this->url_base . '/chat/findContacts/' . $name, [
                'json' => []
            ]);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function sendMessage($name, $text = '', $numberTo = '', $validateNumbers = true)
    {
        try {
            $numberData = $this->formatNumber($numberTo);

            if ($validateNumbers && !$this->validateNumber($name, $numberData)) {
                return false;
            }

            $client = $this->guzzle();
            $body = [
                "number" => $numberData,
                "delay" => 0,
                "linkPreview" => true,
                "text" => $text
            ];

            $response = $client->post($this->url_base . '/message/sendText/' . $name, [
                'json' => $body
            ]);

            return json_decode($response->getBody()->getContents(), true);
        } catch (\Throwable $th) {
            return $this->handleException($th, 'sendMessage');
        }
    }

    public function labels($name)
    {
        $client = $this->guzzle();

        try {
            $response = $client->get($this->url_base . '/label/findLabels/' . $name);
            return json_decode($response->getBody()->getContents(), true);
        } catch (RequestException $e) {
            return $this->handleException($e);
        }
    }

    public function validateNumber($name, $number)
    {
        try {
            $client = $this->guzzle();
            $body = [
                "numbers" => [$number]
            ];

            $response = $client->post($this->url_base . '/chat/whatsappNumbers/' . $name, [
                'json' => $body
            ]);

            $responseBody = json_decode($response->getBody()->getContents(), true);

            if (!empty($responseBody)) {
                foreach ($responseBody as $value) {
                    if ($value['exists'] != true) {
                        return false;
                    }
                }
            }

            return true;
        } catch (\Throwable $th) {
            return $this->handleException($th);
        }
    }

    public function validateHasChat($name, $number)
    {
        try {
            $numberData = $this->formatNumber($number);
            $client = $this->guzzle();
            $body = [
                "where" => [
                    "key" => [
                        "remoteJid" => $numberData
                    ]
                ]
            ];

            $response = $client->post($this->url_base . '/chat/findMessages/' . $name, [
                'json' => $body
            ]);

            $responseBody = json_decode($response->getBody()->getContents(), true);

            if (!empty($responseBody)) {
                if (@$responseBody['messages']['total'] == 0) {
                    return false;
                }
            }

            return true;
        } catch (\Throwable $th) {
            return $this->handleException($th);
        }
    }

    public function sendMedia($name, $text = '', $file = '', $numberTo = '', $type = 'image', $validateNumbers = true)
    {
        try {
            $numberData = $this->formatNumber($numberTo);

            if ($validateNumbers && !$this->validateNumber($name, $numberData)) {
                return false;
            }

            $client = $this->guzzle();
            $body = [
                "number" => $numberData,
                "delay" => 0,
                "mediatype" => $type,
                "caption" => $text,
                "media" => $file,
            ];

            $response = $client->post($this->url_base . '/message/sendMedia/' . $name, [
                'json' => $body
            ]);

            return json_decode($response->getBody()->getContents(), true);
        } catch (\Throwable $th) {
            throw $th;
            die;
            return $this->handleException($th, 'sendMessage');
        }
    }

    public function sendWhatsAppAudio($name, $timer = 100, $file = '', $numberTo = '', $validateNumbers = true)
    {
        try {
            $numberData = $this->formatNumber($numberTo);

            if ($validateNumbers && !$this->validateNumber($name, $numberData)) {
                return false;
            }

            $client = $this->guzzle();
            $body = [
                "number" => $numberData,
                "delay" => 0,
                "encoding" => true,
                "audio" => $file,
            ];

            $response = $client->post($this->url_base . '/message/sendWhatsAppAudio/' . $name, [
                'json' => $body
            ]);

            return json_decode($response->getBody()->getContents(), true);
        } catch (\Throwable $th) {
            return $this->handleException($th, 'sendMessage');
        }
    }

    private function formatNumber($number)
    {
        $numberData = str_replace([" ", "-", "(", ")", "+"], "", $number);
        if (strlen($numberData) === 12 || strlen($numberData) === 13) {
            return $numberData;
        }

        if (substr($numberData, 0, 2) !== "55") {
            if ($numberData[0] == 0) {
                $numberData = substr($numberData, 1);
            }
            $numberData = '55' . $numberData;
        }

        $numberData = preg_replace('/\D/', '', $numberData);
        return $numberData;
    }

    public function guzzle()
    {
        $client = new Client([
            'headers' => [
                'apikey' => $this->global_key
            ],
            'decode_content' => true
        ]);

        return $client;
    }

    private function handleException($e)
    {
        try {
            if (@$e) {
                if (@$e instanceof \GuzzleHttp\Exception\RequestException) {
                    if ($e->hasResponse()) {
                        $responseBody = $e->getResponse()->getBody()->getContents();
                        Log::info("Service Evo API ERROR: " . $responseBody);
                    } else {
                        Log::info("Service Evo API ERROR: " . $e->getMessage());
                    }
                } else {
                    Log::info("Service Evo API ERROR: " . $e->getMessage());
                }
            }
            return false;
        } catch (\Throwable $th) {
            Log::error("Service Evo API handleException ERROR");
            return false;
        }
    }
}
